<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\UserBotTrade;
use App\Models\AiBot;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProcessDailyBotTrades extends Command
{
    protected $signature = 'bots:process-daily';
    protected $description = 'Process daily profit/loss for all active user bot trades, respecting duration and avoiding duplicate logs.';

    public function handle()
    {
        $bots = AiBot::where('status', 'active')->get();

        foreach ($bots as $bot) {
            // 🧠 Check if this bot has already been processed today
            $alreadyProcessed = DB::table('bot_trade_daily_records')
                ->whereDate('recorded_at', now()->toDateString())
                ->whereIn('user_bot_trade_id', function ($query) use ($bot) {
                    $query->select('id')
                        ->from('user_bot_trades')
                        ->where('ai_bot_id', $bot->id);
                })
                ->exists();

            if ($alreadyProcessed) {
                $this->info("⏭ Bot {$bot->name} already processed today. Skipping...");
                continue;
            }

            if ($bot->daily_change === null || $bot->daily_change == 0) {
                $this->info("⏭ No daily change for bot {$bot->name}, skipping.");
                continue;
            }

            DB::transaction(function () use ($bot) {
                $profitPercent = $bot->daily_change;
                $this->info("⚙️ Processing bot {$bot->name} with {$profitPercent}% change...");

                $trades = UserBotTrade::where('ai_bot_id', $bot->id)
                    ->where('status', 'running')
                    ->lockForUpdate()
                    ->get();

                foreach ($trades as $trade) {
                    $now = Carbon::now();
                    $endDate = Carbon::parse($trade->ends_at);

                    // ✅ If trade duration elapsed — complete it
                    if ($now->greaterThanOrEqualTo($endDate)) {
                        $this->info("✅ Trade #{$trade->id} reached end date, marking as completed.");

                        $trade->update([
                            'status' => 'completed',
                            'current_value' => $trade->amount_invested + $trade->profit_earned,
                        ]);

                        // Optional: credit user if profitable
                        if ($trade->current_value > 0 && $trade->user) {
                            $trade->user->increment('profit', $trade->current_value);
                        }

                        DB::table('bot_trade_daily_records')->insert([
                            'user_bot_trade_id' => $trade->id,
                            'percent' => $profitPercent,
                            'profit' => $trade->profit_earned,
                            'recorded_at' => now()->toDateString(),
                            'created_at' => now(),
                            'updated_at' => now(),
                        ]);

                        continue;
                    }

                    // 🧮 Calculate today’s profit or loss (compound)
                    $profitToday = ($trade->current_value * $profitPercent) / 100;

                    // 💰 Update totals
                    $trade->profit_earned += $profitToday;
                    $trade->profit_percent += $profitPercent;
                    $trade->current_value = max(0, $trade->amount_invested + $trade->profit_earned);

                    // 🚨 Stop-loss (auto-close)
                    if ($trade->current_value <= ($trade->amount_invested * 0.1)) {
                        $trade->status = 'auto-closed';
                        $this->info("⚠️ Auto-closing trade #{$trade->id} (value dropped below 10%)");
                    }

                    $trade->save();

                    // 🧾 Log today’s record
                    DB::table('bot_trade_daily_records')->insert([
                        'user_bot_trade_id' => $trade->id,
                        'percent' => $profitPercent,
                        'profit' => $profitToday,
                        'recorded_at' => now()->toDateString(),
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);
                }

                // Update bot’s total earned (for stats)
                $bot->increment('total_earned', $trades->sum(fn($t) => ($t->amount_invested * $profitPercent) / 100));
            });
        }

        $this->info('✅ Daily bot trade processing completed safely (duplicate and duration protected).');
        Log::info('✅ Daily bot trade processing completed safely.');
    }
}
