<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\TradeExpert;
use App\Models\UserCopyTrade;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class ProcessDailyCopyTrades extends Command
{
    protected $signature = 'trading:process-daily';
    protected $description = 'Process daily profit/loss for all active copy trades (updates current value, logs profit/loss, and handles auto-closure).';

    public function handle()
    {
        $experts = TradeExpert::where('status', 'active')->get();

        foreach ($experts as $expert) {
            // 🧠 Skip if no daily change
            if (is_null($expert->daily_change) || $expert->daily_change == 0) {
                $this->info("⏭ No daily change for {$expert->name}, skipping.");
                continue;
            }

            // 🧾 Check if already processed today
            $alreadyProcessed = DB::table('copy_trade_logs')
                ->where('trade_expert_id', $expert->id)
                ->whereDate('created_at', now()->toDateString())
                ->exists();

            if ($alreadyProcessed) {
                $this->info("⏭ {$expert->name} already processed today. Skipping...");
                continue;
            }

            DB::transaction(function () use ($expert) {
                $profitPercent = $expert->daily_change;
                $this->info("⚙️ Processing {$expert->name} ({$profitPercent}% change)...");

                // 🔄 Update expert’s equity (for admin reference)
                $expert->equity = $expert->equity * (1 + ($profitPercent / 100));
                $expert->save();

                // 📊 Fetch all active copy trades for this expert
                $copies = UserCopyTrade::where('trade_expert_id', $expert->id)
                    ->where('status', 'active')
                    ->lockForUpdate()
                    ->get();

                foreach ($copies as $copy) {
                    $user = $copy->user;
                    $oldValue = $copy->current_value;

                    // 🧮 Calculate daily gain/loss
                    $profitToday = ($oldValue * $profitPercent) / 100;
                    $newValue = $oldValue + $profitToday;
                    $profitLoss = $newValue - $copy->amount;
                    $roiPercentage = ($profitLoss / $copy->amount) * 100;

                    // 🚨 Stop-loss (Auto-close if below 10% margin)
                    if ($newValue <= ($copy->amount * 0.1)) {
                        $copy->status = 'auto-closed';
                        $newValue = max(0, $newValue); // prevent negative
                        $this->info("⚠️ Auto-closing trade #{$copy->id} for {$user->name} (value fell below 10%).");
                    }

                    // 💾 Update trade performance
                    $copy->update([
                        'current_value' => $newValue,
                        'profit_loss' => $profitLoss,
                        'roi_percentage' => $roiPercentage,
                        'days_running' => $copy->days_running + 1,
                    ]);

                    // 🧾 Log daily performance (can be + or −)
                    DB::table('copy_trade_logs')->insert([
                        'user_id' => $user->id,
                        'trade_expert_id' => $expert->id,
                        'profit_amount' => $profitToday,
                        'percent_change' => $profitPercent,
                        'note' => $profitPercent >= 0
                            ? "{$expert->name} gained {$profitPercent}% today."
                            : "{$expert->name} lost {$profitPercent}% today.",
                        'created_at' => now(),
                        'updated_at' => now(),
                    ]);

                    $this->info("💹 {$user->name} | {$profitPercent}% | " .
                        ($profitToday >= 0 ? '+' : '-') . number_format(abs($profitToday), 2) .
                        " | New Value: " . number_format($newValue, 2));
                }
            });
        }

        $this->info('✅ Daily copy trade processing completed safely (no user credit, only current value updates).');
        Log::info('✅ Daily copy trade processing completed safely.');
        return 0;
    }
}
