<?php

namespace App\Console\Commands;

use App\Models\User;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Console\Command;
use App\Mail\RealEstatePayoutMail;
use Illuminate\Support\Facades\Log;
use App\Models\RealEstateInvestment;
use Illuminate\Support\Facades\Mail;


class ProcessRealEstatePayouts extends Command
{
    protected $signature = 'investments:payout';
    protected $description = 'Pay matured real estate investments';

    public function handle()
    {
        $investments = RealEstateInvestment::where('status', 'active')
            ->whereDate('end_date', '<=', now())
            ->get();

        foreach ($investments as $inv) {

            $user = $inv->user;
            if (!$user) continue;

            // CREDIT WALLET
            $user->increment('profit', $inv->total_return);

            // UPDATE INVESTMENT
            $inv->update([
                'status' => 'completed',
                'is_paid'     => true,
                'payout_date' => now(),
            ]);

            try {

                // EMAIL
                Mail::to($user->email)
                    ->queue(new RealEstatePayoutMail($user, $inv));

                // IN-APP NOTIFICATION
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Real Estate Investment Paid',
                    'data' => [
                        'message' => "Your real estate investment has been paid successfully.",
                        'investment_id' => $inv->id,
                        'amount_paid' => $inv->total_return,
                        'status' => 'Completed'
                    ],
                ]);
            } catch (\Exception $e) {
                Log::error('Real estate payout notify failed: ' . $e->getMessage());
            }

            $this->info("Paid user {$user->id} for investment {$inv->id}");
        }

        return Command::SUCCESS;
    }
}
