<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\LiveTrade;
use App\Models\UserCopyTrade;
use App\Models\Investment;
use App\Models\UserBotTrade;
use App\Models\Withdrawal;
use App\Models\Deposit;
use Illuminate\Support\Facades\Hash;
use Intervention\Image\Laravel\Facades\Image;
use Intervention\Image\Encoders\JpegEncoder;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use App\Models\User;

class AdminController extends Controller
{
    public function admindashboard(Request $request)
    {
        $pageTitle = 'Admin Dashboard';

        // 🧠 Total users
        $totalUsers = User::where('role', 'user')->count();

        $users = User::where('role', 'user')->latest()->take(6)->get();

        $inactiveUsers = User::where('status', 'Inactive')
            ->where('role', 'user')->count();

        // 💰 Deposits & Withdrawals
        $totalDeposits = Deposit::sum('amount');
        $totalWithdrawals = Withdrawal::sum('amount');

        // 🤖 Latest 5 Bot Trades
        $latestBotTrades = UserBotTrade::with(['user', 'bot'])
            ->latest()
            ->take(5)
            ->get();

        // 💹 Latest 5 Investments
        $latestInvestments = Investment::with('user')
            ->latest()
            ->take(5)
            ->get();

        // 🔁 Latest 5 Copy Trades
        $latestCopyTrades = UserCopyTrade::with('user')
            ->latest()
            ->take(5)
            ->get();

        // ⚡ Latest 5 Live Trades
        $latestLiveTrades = LiveTrade::with('user')
            ->latest()
            ->take(5)
            ->get();

        return view('admin/dashboard', compact(
            'pageTitle',
            'totalUsers',
            'inactiveUsers',
            'totalDeposits',
            'totalWithdrawals',
            'latestBotTrades',
            'latestInvestments',
            'latestCopyTrades',
            'latestLiveTrades',
            'users'
        ));
    }

    //USER MANAGEMENT
    public function adminuserindex(Request $request)
    {
        $pageTitle = 'Manage Users';

        $user = User::where('role', 'user')
            ->orderBy('created_at', 'desc')
            ->paginate(25);

        return view('admin/users/index', compact('pageTitle', 'user'));
    }

    public function adjustUserFunds(Request $request, $userId)
    {
        $validated = $request->validate([
            'type'   => 'required|in:balance,profit,bonus',
            'action' => 'required|in:add,deduct',
            'amount' => 'required|numeric|min:0.01',
            'note'   => 'nullable|string|max:255',
        ]);

        $user = User::findOrFail($userId);
        $type = $validated['type'];
        $amount = $validated['amount'];
        $action = $validated['action'];

        // Calculate new amount
        if ($action === 'add') {
            $user->$type += $amount;
        } else {
            // prevent negative values
            if ($user->$type < $amount) {
                return back()->withErrors(['amount' => ucfirst($type) . ' is too low for deduction.']);
            }
            $user->$type -= $amount;
        }

        $user->save();

        // OPTIONAL: Log admin action
        // TransactionLog::create([
        //     'user_id' => $user->id,
        //     'type'    => $type,
        //     'action'  => $action,
        //     'amount'  => $amount,
        //     'admin_id' => auth()->id(),
        //     'note'    => $validated['note'] ?? null,
        // ]);

        return back()->with('success', "User {$type} updated successfully.");
    }


    public function adminuserdetails(Request $request, $id)
    {
        $user = User::findorfail($id);

        $pageTitle = 'Update ' . $user->username . ' Profile';

        return view('admin/users/details', compact('pageTitle', 'user'));
    }

    public function adminuserupdate($id, Request $request)
    {
        $save = User::find($id);
        $save->status   = trim($request->status);
        $save->alert   = trim($request->alert);
        if (!empty($request->password)) {
            $save->password = Hash::make($request->password);
        }
        $save->save();

        return redirect()->route('admin.user.index')->with('success', 'User Updated successfully.');
    }

    public function adminuserdestroy(Request $request, $id)
    {
        $request = User::find($id);

        $request->delete();

        return redirect()->route('admin.user.index')->with('success', 'User Deleted Successfully');
    }

    public function adminprofile(Request $request)
    {
        return view('admin/profile');
    }


    public function adminprofileupdate(Request $request)
    {
        // ✅ Validate inputs
        $validated = $request->validate([
            'firstname'     => 'required|string|max:255|regex:/^[^<>]*$/',
            'lastname'     => 'required|string|max:255|regex:/^[^<>]*$/',
            'username' => 'required|string|max:255|regex:/^[^<>]*$/',
            'email'    => 'required|email|max:255|unique:users,email,' . Auth::id(),
            'image'    => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        // ✅ Get authenticated user
        $user = Auth::user();

        // ✅ Default to existing image
        $filename = $user->image;

        // ✅ Process new image upload (if provided)
        if ($request->hasFile('image')) {
            $file = $request->file('image');

            // ✅ Generate random filename and always use .jpg
            $randomStr = Str::random(15);
            $filename = $randomStr . '.jpg';

            // ✅ Define destination path
            $destinationPath = public_path('upload/profile/');

            // ✅ Create folder if missing
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            // ✅ Delete old image if it exists
            if ($user->image && file_exists($destinationPath . $user->image)) {
                @unlink($destinationPath . $user->image);
            }

            // ✅ Decode, sanitize, and re-encode to JPG (strips malicious code)
            $image = Image::read($file->getRealPath())
                ->orient()
                ->encode(new JpegEncoder(quality: 85));

            // ✅ Save safely
            $image->save($destinationPath . $filename);
        }

        // ✅ Update user details
        $user->firstname     = trim($validated['firstname']);
        $user->lastname     = trim($validated['lastname']);
        $user->username = trim($validated['username']);
        $user->email    = trim($validated['email']);
        $user->image    = $filename;

        $user->save();

        // ✅ Success message
        return redirect()->back()->with('success', 'Profile updated successfully!');
    }
}
