<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;


class ApiController extends Controller
{
    public function apimanagement()
    {
        $pageTitle = 'API Key Management';

        $apis = [
            [
                'name' => 'Twelve Data',
                'key' => env('TWELVEDATA_KEY'),
                'description' => 'Used for fetching stock and forex market data.',
            ],
            [
                'name' => 'Alpha Vantage',
                'key' => env('ALPHA_VANTAGE'),
                'description' => 'Used for commodities and bond market data.',
            ],
            [
                'name' => 'Logo.dev (Publishable Key)',
                'key' => env('LOGODEV_PUBLISHABLE_KEY'),
                'description' => 'Used to fetch company logos from logo.dev (public key).',
            ],
            [
                'name' => 'Logo.dev (Secret Key)',
                'key' => env('LOGODEV_SECRET_KEY'),
                'description' => 'Used for backend authentication and secure API calls (keep private).',
            ],
        ];

        return view('admin.api_management', compact('pageTitle', 'apis'));
    }


    public function apiupdate(Request $request)
    {
        $request->validate([
            'TWELVEDATA_KEY' => 'nullable|string|max:255',
            'ALPHA_VANTAGE' => 'nullable|string|max:255',
            'LOGODEV_PUBLISHABLE_KEY' => 'nullable|string|max:255',
            'LOGODEV_SECRET_KEY' => 'nullable|string|max:255',
        ]);

        $envPath = base_path('.env');
        $envContent = File::get($envPath);

        foreach (
            $request->only([
                'TWELVEDATA_KEY',
                'ALPHA_VANTAGE',
                'LOGODEV_PUBLISHABLE_KEY',
                'LOGODEV_SECRET_KEY'
            ]) as $key => $value
        ) {
            // Safely replace or append key=value
            if (str_contains($envContent, $key . '=')) {
                $envContent = preg_replace(
                    "/^{$key}=.*/m",
                    "{$key}=\"{$value}\"",
                    $envContent
                );
            } else {
                $envContent .= "\n{$key}=\"{$value}\"";
            }
        }

        File::put($envPath, $envContent);

        // Reload environment vars immediately
        Artisan::call('config:clear');
        Artisan::call('cache:clear');
        Artisan::call('config:cache');

        return back()->with('success', 'API keys updated successfully!');
    }
}
