<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Models\WebsiteSetting;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use App\Mail\VerificationCodeMail;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): View
    {
        $pageTitle = 'Login';

        return view('auth.login', compact('pageTitle'));
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request): RedirectResponse
    {
        $request->authenticate();
        $request->session()->regenerate();

        $user = $request->user();

        // 🚫 Restrict admin from logging in here
        if ($user->role === 'admin') {
            Auth::logout();
            return redirect()->route('admin.login')
                ->withErrors(['email' => 'Admins must log in through the admin portal.']);
        }

        // 🔧 Fetch setting: is email verification required?
        $requireVerification = WebsiteSetting::value('require_email_verification') ?? true;

        // 🚨 Handle email verification based on setting
        if ($requireVerification) {

            // ❗ Email verification REQUIRED
            if (is_null($user->email_verified_at)) {
                // Generate new verification code
                $newCode = rand(100000, 999999);
                $user->update(['verification_code' => $newCode]);

                try {
                    Mail::to($user->email)->send(new VerificationCodeMail($user));
                    Log::info('Verification code resent during login', [
                        'email' => $user->email,
                        'code' => $newCode
                    ]);
                } catch (\Exception $e) {
                    Log::error('Failed to send verification code during login', [
                        'email' => $user->email,
                        'error' => $e->getMessage()
                    ]);
                }

                Auth::logout();

                session(['unverified_email' => $user->email]);

                return redirect()->route('verify.email.page')
                    ->with('warning', 'Please verify your email before logging in. A new verification code has been sent to your email.')
                    ->with('email', $user->email);
            }
        } else {

            // ❗ Email verification NOT required → auto verify if not already verified
            if (is_null($user->email_verified_at)) {
                $user->update([
                    'email_verified_at' => now(),
                    'verification_code' => null,
                ]);

                Log::info('Email auto-verified during login because verification is disabled', [
                    'user' => $user->id,
                    'email' => $user->email
                ]);
            }
        }

        // 🔐 If 2FA enabled → go to 2FA screen
        if ($user->google2fa_secret) {
            session(['2fa:user:id' => $user->id]);
            return redirect()->route('2fa.verify.form');
        }

        // 📌 Update login metadata
        $user->update([
            'last_login_at'     => now(),
            'last_seen_at'      => now(),
            'last_login_ip'     => $request->ip(),
            'last_login_device' => $request->header('User-Agent'),
        ]);

        logUserActivity('Login', 'Account Login');

        return redirect()->intended(route('user.dashboard'))
            ->with('success', 'Login successful! Welcome back.');
    }


    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }


    protected function authenticated(Request $request, $user)
    {
        if ($user->role === 'admin') {
            Auth::logout();
            return redirect()->route('admin.login')->withErrors(['email' => 'Admins must log in through the admin portal.']);
        }

        return redirect()->route('user.dashboard');
    }
}
