<?php

namespace App\Http\Controllers;

use App\Models\KycVerification;
use App\Models\Notification;
use App\Models\User;
use App\Mail\KycRejectedMail;
use App\Mail\KycApprovedMail;
use Illuminate\Support\Facades\Log;
use App\Mail\KycAdminAlertMail;
use Illuminate\Support\Facades\Mail;
use App\Mail\KycSubmittedMail;
use App\Models\WebsiteSetting;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use Intervention\Image\Laravel\Facades\Image;
use Intervention\Image\Encoders\JpegEncoder;

class KycVerificationController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function startkyc(Request $request)
    {
        // dd($request->all());

        $request->validate([
            'id_type' => 'required|string|max:255',
            'firstname' => 'required|string|max:255',
            'lastname' => 'required|string|max:255',
            'email' => 'required|email',
            'phone' => 'required|string|max:20',
            'dob' => 'required|date',
            'social_username' => 'nullable|string|max:255',
            'country' => 'required|string|max:255',
            'state' => 'required|string|max:255',
            'city' => 'required|string|max:255',
            'address' => 'required|string|max:500',
            'front_img' => 'required|image|mimes:jpeg,png,jpg|max:2048',
            'back_img' => 'required|image|mimes:jpeg,png,jpg|max:2048',
        ]);

        $kyc = new KycVerification();
        $kyc->user_id = Auth::id();
        $kyc->id_type = $request->id_type;
        $kyc->firstname = $request->firstname;
        $kyc->lastname = $request->lastname;
        $kyc->email = $request->email;
        $kyc->phone = $request->phone;
        $kyc->dob = $request->dob;
        $kyc->social_username = $request->social_username;
        $kyc->country = $request->country;
        $kyc->state = $request->state;
        $kyc->city = $request->city;
        $kyc->address = $request->address;
        $kyc->status = 'Pending';

        if ($request->hasFile('front_img')) {
            $file = $request->file('front_img');

            // ✅ Generate random filename and always use .jpg
            $randomStr = Str::random(30);
            $filename = $randomStr . '.jpg';

            // ✅ Define destination path
            $destinationPath = public_path('upload/kyc/');

            // ✅ Create folder if missing
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            // ✅ Decode, sanitize, and re-encode to JPG (strips malicious code)
            $image = Image::read($file->getRealPath())
                ->orient()
                ->encode(new JpegEncoder(quality: 85));

            // ✅ Save safely
            $image->save($destinationPath . $filename);

            // ✅ Assign to model
            $kyc->front_img = $filename;
        }

        if ($request->hasFile('back_img')) {
            $file = $request->file('back_img');

            // ✅ Generate random filename and always use .jpg
            $randomStr = Str::random(30);
            $filename = $randomStr . '.jpg';

            // ✅ Define destination path
            $destinationPath = public_path('upload/kyc/');

            // ✅ Create folder if missing
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            // ✅ Decode, sanitize, and re-encode to JPG (strips malicious code)
            $image = Image::read($file->getRealPath())
                ->orient()
                ->encode(new JpegEncoder(quality: 85));

            // ✅ Save safely
            $image->save($destinationPath . $filename);

            // ✅ Assign to model
            $kyc->back_img = $filename;
        }

        $kyc->save();

        // ✅ Send confirmation to user
        try {
            Mail::to($kyc->email)->queue(new KycSubmittedMail($kyc));
        } catch (\Exception $e) {
            Log::error('Failed to send KYC user email: ' . $e->getMessage());
        }

        // ✅ Notify admin
        try {
            $adminEmail = WebsiteSetting::value('site_email');
            if ($adminEmail) {
                Mail::to($adminEmail)->queue(new KycAdminAlertMail($kyc));
            }
        } catch (\Exception $e) {
            Log::error('Failed to send KYC admin email: ' . $e->getMessage());
        }

        /** ==========================
         *  ✅ CREATE NOTIFICATIONS
         *  ========================== */

        // 🔹 Notify Admin
        $admin = User::where('role', 'admin')->first();
        if ($admin) {
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => 'App\Models\User',
                'notifiable_id' => $admin->id,
                'type' => 'New KYC Submission',
                'data' => json_encode([
                    'message' => "New KYC submission from {$kyc->firstname} {$kyc->lastname} ({$kyc->email})",
                    'kyc_id' => $kyc->id,
                    'status' => 'Pending',
                ]),
            ]);
        }

        // 🔹 Notify User
        Notification::create([
            'id' => Str::uuid(),
            'notifiable_type' => 'App\Models\User',
            'notifiable_id' => $kyc->user_id,
            'type' => 'KYC Submitted',
            'data' => json_encode([
                'message' => 'Your KYC verification has been submitted and is now pending review.',
                'kyc_id' => $kyc->id,
                'status' => 'Pending',
            ]),
        ]);

        return redirect()
            ->route('user.dashboard')
            ->with('success', 'KYC verification submitted successfully! Status: Pending');
    }

    /**
     * Display the specified resource.
     */
    public function adminkycindex(Request $request)
    {
        $pageTitle = 'User Kyc List';

        $kyc = KycVerification::paginate(25);

        return view('admin/kyc/index', compact('pageTitle', 'kyc'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function adminkycdetails(Request $request, $id)
    {
        $kyc = KycVerification::findorfail($id);

        return view('admin/kyc/details', compact('kyc'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function updatekycStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:Pending,Approved,Rejected',
        ]);

        $kyc = KycVerification::findOrFail($id);

        $kyc->status = $request->status;

        $kyc->save();

        $user = $kyc->user;
        $admin = User::where('role', 'admin')->first();

        try {
            if ($kyc->status === 'Approved') {
                Mail::to($kyc->email)->queue(new KycApprovedMail($kyc));
            } elseif ($kyc->status === 'Rejected') {
                Mail::to($kyc->email)->queue(new KycRejectedMail($kyc));
            }
        } catch (\Exception $e) {
            Log::error('KYC status email failed: ' . $e->getMessage());
        }

        // 🔹 Notify User
        $userMessage = match ($kyc->status) {
            'Approved' => 'Your KYC verification has been approved. You now have full account access.',
            'Rejected' => 'Your KYC verification was rejected. Please review and resubmit your documents.',
            default => 'Your KYC status was updated.',
        };

        Notification::create([
            'id' => Str::uuid(),
            'notifiable_type' => 'App\Models\User',
            'notifiable_id' => $kyc->user_id,
            'type' => "KYC {$kyc->status}",
            'data' => json_encode([
                'message' => $userMessage,
                'kyc_id' => $kyc->id,
                'status' => $kyc->status,
            ]),
        ]);

        // 🔹 Notify Admin (optional audit trail)
        if ($admin) {
            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => 'App\Models\User',
                'notifiable_id' => $admin->id,
                'type' => "KYC {$kyc->status} Update",
                'data' => json_encode([
                    'message' => "{$kyc->firstname} {$kyc->lastname}'s KYC has been updated to {$kyc->status}.",
                    'kyc_id' => $kyc->id,
                ]),
            ]);
        }

        return redirect()->back()->with('success', "✅ KYC status updated to {$kyc->status}");
    }
}
