<?php

namespace App\Http\Controllers;

use App\Models\LoanType;
use App\Mail\LoanApprovedMail;
use App\Mail\LoanRejectedMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;
use App\Models\User;
use Illuminate\Support\Str;
use App\Models\Notification;
use Illuminate\Support\Facades\Auth;
use App\Models\Loan;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class LoanController extends Controller
{
    public function adminloantype(Request $request)
    {
        $pageTitle = 'Loan Types';

        $type = LoanType::paginate(15);

        return view('admin/loan/type', compact('type', 'pageTitle'));
    }

    public function adminloanindex(Request $request)
    {
        $data = Loan::latest()->paginate(25);

        return view('admin/loan/application', compact('data'));
    }

    public function adminuserloans(Request $request, $id)
    {
        $data = Loan::findorfail($id);

        return view('admin/loan/user_loan', compact('data'));
    }

    public function updateLoanStatus(Request $request, $id)
    {
        $request->validate([
            'status' => 'required|in:approved,rejected',
        ]);

        $loan = Loan::findOrFail($id);
        $user = $loan->user;

        if ($loan->status !== 'pending') {
            return back()->with('error', 'This loan has already been processed.');
        }

        // ✅ Update status
        $loan->status = $request->status;
        $loan->save();

        // 🔹 Common data
        $amountFormatted = '$' . number_format($loan->amount, 2);
        $loanType = $loan->loanType->name ?? 'Loan';
        // 🧾 If APPROVED
        if ($request->status === 'approved') {

            // 💰 Credit user's balance
            $user->balance += $loan->amount;
            $user->save();


            // 📩 Send email to user
            try {
                Mail::to($user->email)->queue(new LoanApprovedMail($loan));
            } catch (\Exception $e) {
                Log::error("LoanApprovedMail failed: " . $e->getMessage());
            }

            // 🔔 Create Notifications
            try {
                // User notification
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Loan Approved',
                    'data' => [
                        'message' => "Your {$loanType} of {$amountFormatted} has been approved and credited to your account.",
                        'loan_id' => $loan->id,
                        'status' => 'Approved',
                    ],
                ]);
            } catch (\Exception $e) {
                Log::error('Loan approval notification failed: ' . $e->getMessage());
            }

            return back()->with('success', '✅ Loan approved and funds credited successfully.');
        }

        // 🛑 If REJECTED
        if ($request->status === 'rejected') {

            // 📩 Send rejection email
            try {
                Mail::to($user->email)->queue(new LoanRejectedMail($loan));
            } catch (\Exception $e) {
                Log::error("LoanRejectedMail failed: " . $e->getMessage());
            }
            // 🔔 Create Notifications
            try {
                // User notification
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Loan Rejected',
                    'data' => [
                        'message' => "Your {$loanType} of {$amountFormatted} has been rejected. Please contact support for more info.",
                        'loan_id' => $loan->id,
                        'status' => 'Rejected',
                    ],
                ]);
            } catch (\Exception $e) {
                Log::error('Loan rejection notification failed: ' . $e->getMessage());
            }

            return back()->with('info', '❌ Loan marked as rejected.');
        }
    }


    public function adminloanedittype(Request $request, $id)
    {
        $type = LoanType::findorfail($id);

        $pageTitle = 'Manage ' . $type->name . ' Type';

        return view('admin/loan/type_edit', compact('type', 'pageTitle'));
    }

    public function adminloanstoretype(Request $request)
    {
        $type = new LoanType;

        $type->name = $request->name;
        $type->emoji = $request->emoji;
        $type->description = $request->description;
        $type->min_duration = $request->min_duration;
        $type->max_duration = $request->max_duration;
        $type->min_amount = $request->min_amount;
        $type->max_amount = $request->max_amount;
        $type->interest_rate = $request->interest_rate;

        $type->save();

        return redirect()->route('admin.loan.type')->with('success', 'Loan Type addedd successfully!');
    }

    public function adminloanupdatetype(Request $request, $id)
    {
        $type = LoanType::findorfail($id);

        $type->name = $request->name;
        $type->emoji = $request->emoji;
        $type->description = $request->description;
        $type->min_duration = $request->min_duration;
        $type->max_duration = $request->max_duration;
        $type->min_amount = $request->min_amount;
        $type->max_amount = $request->max_amount;
        $type->interest_rate = $request->interest_rate;

        $type->save();

        return redirect()->route('admin.loan.type')->with('success', 'Loan Type updated successfully!');
    }



    //USER
    public function showuserloanapply(Request $request)
    {
        $pageTitle = 'Loan Application';

        $type = LoanType::all();

        return view('user/apply_loan', compact('pageTitle', 'type'));
    }


    public function storeuserloan(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'loan_type_id'   => 'required|exists:loan_types,id',
            'monthly_income' => 'required|numeric|min:100',
            'amount'         => 'required|numeric|min:100',
            'duration'       => 'required|integer|min:1|max:60',
            'reason'         => 'required|string|max:500',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors()
            ], 422); // 422: Unprocessable Entity (used for validation errors)
        }

        $user = Auth::user();
        $loanType = LoanType::findOrFail($request->loan_type_id);

        // 🧮 Validate amount range
        if ($request->amount < $loanType->min_amount || $request->amount > $loanType->max_amount) {
            return back()->with('error', "Amount must be between $" . number_format($loanType->min_amount) . " and $" . number_format($loanType->max_amount) . " for this loan type.");
        }

        // 🕒 Validate duration range
        if ($request->duration < $loanType->min_duration || $request->duration > $loanType->max_duration) {
            return back()->with('error', "Duration must be between {$loanType->min_duration} and {$loanType->max_duration} months for this loan type.");
        }

        // 💸 Eligibility check (max 3× monthly income)
        if ($request->amount > ($request->monthly_income * 3)) {
            return back()->with('error', 'Requested amount exceeds your income-based eligibility limit (max 3× monthly income).');
        }

        // 💰 Use loan type’s fixed interest rate
        $interestRate = $loanType->interest_rate;

        // 🧮 Simple interest formula
        $totalInterest  = ($request->amount * $interestRate / 100) * ($request->duration / 12);
        $totalPayable   = $request->amount + $totalInterest;
        $monthlyPayment = $totalPayable / $request->duration;

        // 💾 Create loan record
        Loan::create([
            'user_id'         => $user->id,
            'loan_type_id'    => $loanType->id,
            'amount'          => $request->amount,
            'duration'        => $request->duration,
            'interest_rate'   => $interestRate,
            'total_payable'   => round($totalPayable, 2),
            'monthly_payment' => round($monthlyPayment, 2),
            'monthly_income'  => $request->monthly_income,
            'reason'          => $request->reason,
            'status'          => 'pending',
        ]);

        return redirect()->route('user.apply_loan')->with('success', '✅ Loan application submitted successfully!');
    }


    public function showuserloanhistory(Request $request)
    {
        $pageTitle = 'Loan History';
        $user = Auth::user();

        // ✅ Count all loan applications
        $totalLoans = Loan::where('user_id', $user->id)->count();

        // ✅ Count approved and pending loans
        $approvedLoans = Loan::where('user_id', $user->id)
            ->where('status', 'approved')
            ->count();

        $pendingLoans = Loan::where('user_id', $user->id)
            ->where('status', 'pending')
            ->count();

        // ✅ Sum total value of all loans (regardless of status)
        $totalLoanValue = Loan::where('user_id', $user->id)->sum('amount');

        // ✅ Optionally: Fetch loan records for display
        $loans = Loan::where('user_id', $user->id)
            ->latest()
            ->paginate(10); // Optional pagination

        return view('user.loan_history', compact(
            'pageTitle',
            'totalLoans',
            'approvedLoans',
            'pendingLoans',
            'totalLoanValue',
            'loans'
        ));
    }
}
