<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\Deposit;
use Illuminate\Support\Facades\Log;
use App\Mail\DepositPendingMail;
use App\Mail\NewDepositAlertMail;
use App\Mail\PaypalDepositApprovedMail;
use Illuminate\Support\Facades\Mail;
use App\Models\WebsiteSetting;
use App\Models\User;
use Illuminate\Support\Str;
use App\Models\Notification;
use Srmklive\PayPal\Services\PayPal as PayPalClient;


class PayPalController extends Controller
{
    public function checkout(Request $request)
    {
        $provider = new PayPalClient();
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();


        $response = $provider->createOrder([
            "intent" => "CAPTURE",
            "application_context" => [
                "return_url" => route('paypal.success'),
                "cancel_url" => route('paypal.cancel'),
            ],
            "purchase_units" => [
                [
                    "amount" => [
                        "currency_code" => config('paypal.currency', 'USD'),
                        "value" => number_format($request->amount, 2, '.', ''),
                    ],
                ],
            ],
        ]);

        if (isset($response['id']) && $response['id'] != null) {
            foreach ($response['links'] as $link) {
                if ($link['rel'] === 'approve') {
                    return redirect()->away($link['href']);
                }
            }
        }

        return redirect()->route('paypal.cancel')->with('error', 'Something went wrong with PayPal.');
    }

    public function success(Request $request)
    {
        $provider = new PayPalClient();
        $provider->setApiCredentials(config('paypal'));
        $provider->getAccessToken();

        $token = $request->token ?? $request->orderID ?? null;
        $response = $provider->capturePaymentOrder($token);

        \Log::info('PayPal Capture Response:', $response);

        if (isset($response['status']) && $response['status'] === 'COMPLETED') {
            $capture = $response['purchase_units'][0]['payments']['captures'][0] ?? null;
            $amount  = $capture['amount']['value'] ?? 0;
            $currency = $capture['amount']['currency_code'] ?? 'USD';
            $transactionId = $capture['id'] ?? '';

            // ✅ Identify PayPal gateway
            $paypalGatewayId = 6;

            // ✅ Create deposit record
            $deposit = Deposit::create([
                'user_id' => Auth::id(),
                'payment_gateway_id' => $paypalGatewayId,
                'amount' => $amount,
                'currency' => $currency,
                'remark' => 'PayPal Deposit - Transaction ' . $transactionId,
                'status' => 'approved', // PayPal is instant
                'pay_image' => null,
            ]);

            $user = $deposit->user;
            $admin = User::where('role', 'admin')->first();

            // ✅ Credit the user's balance
            try {
                $user->balance = ($user->balance ?? 0) + $deposit->amount;
                $user->save();

                Log::info("✅ PayPal Deposit credited to user {$user->id}. New balance: {$user->balance}");
            } catch (\Exception $e) {
                Log::error("❌ Failed to credit user balance for PayPal deposit: " . $e->getMessage());
            }

            // ✅ Activity Log (user)
            if (function_exists('logUserActivity')) {
                logUserActivity('Deposit', 'You made a PayPal deposit of $' . number_format($deposit->amount, 2));
            }

            // ✅ Send email to user
            try {
                Mail::to($user->email)->queue(new PaypalDepositApprovedMail($deposit));
            } catch (\Exception $e) {
                Log::error('DepositApprovedMail failed: ' . $e->getMessage());
            }

            // ✅ Send email to admin
            try {
                $adminEmail = WebsiteSetting::value('site_email');
                if ($adminEmail) {
                    Mail::to($adminEmail)->queue(new NewDepositAlertMail($deposit));
                }
            } catch (\Exception $e) {
                Log::error('NewDepositAlertMail failed: ' . $e->getMessage());
            }

            // ✅ Notifications
            try {
                if ($admin) {
                    Notification::create([
                        'id' => Str::uuid(),
                        'notifiable_type' => User::class,
                        'notifiable_id' => $admin->id,
                        'type' => 'New Deposit',
                        'data' => [
                            'message' => "{$user->firstname} {$user->lastname} made a PayPal deposit of {$deposit->amount}",
                            'deposit_id' => $deposit->id,
                            'status' => 'Approved',
                        ],
                    ]);
                }

                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $user->id,
                    'type' => 'Deposit Successful',
                    'data' => [
                        'message' => "Your PayPal deposit of {$deposit->amount} has been processed successfully.",
                        'deposit_id' => $deposit->id,
                        'status' => 'Approved',
                    ],
                ]);
            } catch (\Exception $e) {
                Log::error('Deposit notifications failed: ' . $e->getMessage());
            }

            return redirect()->route('user.statement')
                ->with('success', 'PayPal deposit completed successfully!');
        }

        return redirect()->route('user.statement')
            ->with('error', 'Payment failed or was not approved.');
    }


    public function cancel()
    {
        return redirect()->route('user.dashboard')->with('error', 'You cancelled the payment.');
    }
}
