<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use App\Models\RealEstateInvestment;
use App\Models\RealEstateImage;
use Illuminate\Support\Str;
use App\Models\RealEstate;
use Illuminate\Support\Facades\Validator;

class RealEstateController extends Controller
{
    //USER REAL ESTATE

    public function showuserreal(Request $request)
    {
        $user = Auth::user();

        $pageTitle = 'Real Estate Investment';

        $openreal = RealEstate::where('status', 'open')->get();

        $closereal = RealEstate::where('status', 'closed')->get();

        $invest = RealEstateInvestment::where('user_id', $user->id)
            ->latest()
            ->paginate(10);

        // ✅ Total number of investments
        $totalInvestments = RealEstateInvestment::where('user_id', $user->id)->count();

        // ✅ Count active investments
        $activeInvestments = RealEstateInvestment::where('user_id', $user->id)
            ->where('status', 'active')
            ->count();

        // ✅ Count completed investments
        $completedInvestments = RealEstateInvestment::where('user_id', $user->id)
            ->where('status', 'completed')
            ->count();

        // ✅ Sum total invested amount
        $totalAmountInvested = RealEstateInvestment::where('user_id', $user->id)->sum('amount');

        return view('user.real_estate', compact(
            'pageTitle',
            'openreal',
            'closereal',
            'invest',
            'totalInvestments',
            'activeInvestments',
            'completedInvestments',
            'totalAmountInvested'
        ));
    }


    public function showuserrealdetails(RealEstate $real)
    {
        $real->load('images');

        $pageTitle =  $real->title;

        return view('user.realestate_details', compact('pageTitle', 'real'));
    }

    public function realestateinvestdetails(Request $request, $id)
    {
        $invest = RealEstateInvestment::findorfail($id);

        $pageTitle = 'View ' . $invest->estate->title;

        return view('user.estate_invest_details', compact('invest', 'pageTitle'));
    }




    //ADMIN REAL ESTATE

    public function adminrealindex(Request $request)
    {
        $pageTitle = 'Real Estates';

        $real = RealEstate::latest()->paginate(25);

        return view('admin.realestate.index', compact('pageTitle', 'real'));
    }

    public function adminrealadd(Request $request)
    {
        $pageTitle = 'Add Real Estates';

        return view('admin.realestate.add', compact('pageTitle'));
    }


    public function adminrealstore(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'price' => 'required|numeric',
            'roi' => 'nullable|numeric',
            'stories' => 'nullable|numeric',
            'status' => 'required',
            'end_date' => 'required',
            'strategy' => 'required',
            'type' => 'required',
            'objective' => 'nullable',
            'why_project' => 'nullable',
            'why_sponsor' => 'nullable',
            'short_desc' => 'nullable',
            'description' => 'nullable',

            'site_plan' => 'nullable|mimes:pdf|max:20240',
            'organizational_chart' => 'nullable|mimes:pdf|max:20240',
            'sponsor_record' => 'nullable|mimes:pdf|max:20240',

            'images.*' => 'image|mimes:jpg,jpeg,png|max:5120',
        ]);

        $data = $request->except(['images']);

        $docs = ['site_plan', 'organizational_chart', 'sponsor_record'];

        foreach ($docs as $doc) {
            if ($request->hasFile($doc)) {
                $file = $request->file($doc);
                $name = Str::random(12) . '.' . $file->getClientOriginalExtension();

                $path = public_path('uploads/realestate/docs');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }

                $file->move($path, $name);

                $data[$doc] = 'uploads/realestate/docs/' . $name;
            }
        }

        $estate = RealEstate::create($data);

        if ($request->hasFile('images')) {

            foreach ($request->file('images') as $file) {

                $name = Str::random(12) . '.' . $file->getClientOriginalExtension();

                $path = public_path('uploads/realestate/images');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }

                $file->move($path, $name);

                RealEstateImage::create([
                    'real_estate_id' => $estate->id,
                    'image' => 'uploads/realestate/images/' . $name
                ]);
            }
        }

        return redirect()
            ->route('admin.realestate.index')
            ->with('success', 'Real estate project created successfully.');
    }

    public function adminrealedit(request $request, $id)
    {
        $estate = RealEstate::findOrFail($id);

        $pageTitle = 'Manage ' . $estate->title;

        return view('admin.realestate.edit', compact('estate', 'pageTitle'));
    }


    public function adminrealupdate(Request $request, $id)
    {
        $estate = RealEstate::findOrFail($id);

        $request->validate([
            'title' => 'required|string|max:255',
            'price' => 'required|numeric',
            'roi' => 'nullable|numeric',
            'stories' => 'nullable|numeric',
            'status' => 'required',
            'end_date' => 'required',
            'strategy' => 'required',
            'type' => 'required',
            'objective' => 'nullable',
            'why_project' => 'nullable',
            'why_sponsor' => 'nullable',
            'short_desc' => 'nullable',
            'description' => 'nullable',

            'site_plan' => 'nullable|mimes:pdf|max:20240',
            'organizational_chart' => 'nullable|mimes:pdf|max:20240',
            'sponsor_record' => 'nullable|mimes:pdf|max:20240',

        ]);

        $data = $request->except(['images']);

        $docs = ['site_plan', 'organizational_chart', 'sponsor_record'];

        foreach ($docs as $doc) {
            if ($request->hasFile($doc)) {

                // delete old file if exists
                if ($estate->$doc && file_exists(public_path($estate->$doc))) {
                    unlink(public_path($estate->$doc));
                }

                $file = $request->file($doc);
                $name = Str::random(12) . '.' . $file->getClientOriginalExtension();

                $path = public_path('uploads/realestate/docs');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }

                $file->move($path, $name);

                $data[$doc] = 'uploads/realestate/docs/' . $name;
            }
        }

        $estate->update($data);

        if ($request->hasFile('images')) {

            foreach ($request->file('images') as $file) {

                $name = Str::random(12) . '.' . $file->getClientOriginalExtension();

                $path = public_path('uploads/realestate/images');
                if (!file_exists($path)) {
                    mkdir($path, 0755, true);
                }

                $file->move($path, $name);

                RealEstateImage::create([
                    'real_estate_id' => $estate->id,
                    'image' => 'uploads/realestate/images/' . $name
                ]);
            }
        }

        return redirect()->route('admin.realestate.index')->with('success', 'Real estate project updated successfully.');
    }


    public function adminrealdestroy(RealEstate $realEstate)
    {
        $realEstate->delete();
        return back()->with('success', 'Deleted');
    }


    public function adminuserestate(Request $request)
    {
        $pageTitle = 'Real Estate Investments';

        $trade = RealEstateInvestment::paginate(30);

        return view('admin.realestate.trades', compact('trade', 'pageTitle'));
    }
}
