<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use App\Mail\RealEstateInvestmentUserMail;
use App\Mail\RealEstateInvestmentAdminMail;
use Illuminate\Support\Facades\Log;
use App\Models\Notification;
use Illuminate\Support\Str;
use App\Models\User;
use App\Models\RealEstate;
use Illuminate\Support\Facades\Auth;
use App\Models\RealEstateInvestment;

class RealEstateInvestmentController extends Controller
{
    public function investrealestate(Request $request, $id)
    {
        $estate = RealEstate::findOrFail($id);

        if ($estate->status !== 'open') {
            return back()->with('error', 'Project is closed.');
        }

        $request->validate([
            'amount' => 'required|numeric|min:' . $estate->price,
            'duration_days' => 'required|integer|min:1'
        ]);

        $user = Auth::user();

        if ($user->balance < $request->amount) {
            return back()->with('error', 'Insufficient wallet balance.');
        }

        $amount = $request->amount;
        $duration = (int) $request->duration_days;
        $roiPercent = $estate->roi;

        // MATCH FRONTEND FORMULA
        $profit = $amount * ($roiPercent / 100) * ($duration / 30);
        $totalReturn = $amount + $profit;

        $investment = RealEstateInvestment::create([
            'user_id' => $user->id,
            'real_estate_id' => $estate->id,
            'amount' => $amount,
            'duration_days' => $duration,
            'roi_percent' => $roiPercent,
            'profit' => $profit,
            'total_return' => $totalReturn,
            'start_date' => now(),
            'end_date' => now()->addDays($duration),
            'status' => 'active'
        ]);

        $user->decrement('balance', $amount);


        $admin = User::where('role', 'admin')->first();

        try {

            Mail::to($user->email)
                ->send(new RealEstateInvestmentUserMail($user, $estate, $investment));

            if ($admin) {
                Mail::to($admin->email)
                    ->send(new RealEstateInvestmentAdminMail($user, $estate, $investment));
            }

            if ($admin) {
                Notification::create([
                    'id' => Str::uuid(),
                    'notifiable_type' => User::class,
                    'notifiable_id' => $admin->id,
                    'type' => 'New Real Estate Investment',
                    'data' => [
                        'message' => "{$user->firstname} {$user->lastname} invested in {$estate->title}",
                        'estate_id' => $estate->id,
                        'amount' => $amount,
                        'duration' => $duration . ' days',
                        'status' => 'Active',
                    ],
                ]);
            }

            Notification::create([
                'id' => Str::uuid(),
                'notifiable_type' => User::class,
                'notifiable_id' => $user->id,
                'type' => 'Investment Started',
                'data' => [
                    'message' => "Your investment in {$estate->title} has started successfully.",
                    'estate_id' => $estate->id,
                    'amount' => $amount,
                    'duration' => $duration . ' days',
                    'expected_return' => $totalReturn,
                    'status' => 'Active',
                ],
            ]);
        } catch (\Exception $e) {
            Log::error($e->getMessage());
        }

        return back()->with('success', 'Investment started successfully.');
    }
}
