<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Str;
use App\Models\TradeExpert;
use App\Models\UserCopyTrade;
use Illuminate\Support\Facades\Validator;

class TradeExpertController extends Controller
{
    public function expertindex(Request $request)
    {
        $pageTitle = 'Trade Experts';

        $experts = TradeExpert::latest()->paginate(20);

        return view('admin.experts.index', compact('experts', 'pageTitle'));
    }

    /**
     * Show create form
     */
    public function expertcreate()
    {
        $pageTitle = 'Create Trade Experts';

        return view('admin.experts.create', compact('pageTitle'));
    }

    /**
     * Store new expert
     */
    public function expertstore(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'status' => 'required|in:active,inactive',
            'rating' => 'nullable|numeric|min:0|max:5',
            'followers' => 'nullable|integer|min:0',
            'win_rate' => 'nullable|numeric|min:0|max:100',
            'total_return' => 'nullable|numeric|min:0',
            'total_trades' => 'nullable|integer|min:0',
            'min_amount' => 'required|numeric|min:0',
            'equity' => 'required|numeric|min:0',
            'daily_change' => 'required|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        $exp = new TradeEXpert;
        $exp->name = $request->name;
        $exp->title = $request->title;
        $exp->status = $request->status;
        $exp->rating = $request->rating;
        $exp->followers = $request->followers;
        $exp->win_rate = $request->win_rate;
        $exp->total_return = $request->total_return;
        $exp->total_trades = $request->total_trades;
        $exp->min_amount = $request->min_amount;
        $exp->equity = $request->equity;
        $exp->daily_change = $request->daily_change;
        $exp->description = $request->description;

        if (!empty($request->file('image'))) {

            $file = $request->file('image');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/experts/', $filename);
            $exp->image = $filename;
        }

        $exp->save();

        return redirect()->route('admin.experts.index')
            ->with('success', 'Trade Expert created successfully!');
    }

    /**
     * Edit expert
     */
    public function expertsedit($id)
    {
        $expert = TradeExpert::findOrFail($id);

        $pageTitle = 'Update ' . $expert->name . ' Details';

        return view('admin.experts.edit', compact('expert', 'pageTitle'));
    }

    /**
     * Update expert
     */
    public function expertupdate(Request $request, $id)
    {

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'title' => 'nullable|string|max:255',
            'image' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            'status' => 'required|in:active,inactive',
            'rating' => 'nullable|numeric|min:0|max:5',
            'followers' => 'nullable|integer|min:0',
            'win_rate' => 'nullable|numeric|min:0|max:100',
            'total_return' => 'nullable|numeric|min:0',
            'total_trades' => 'nullable|integer|min:0',
            'min_amount' => 'required|numeric|min:0',
            'equity' => 'required|numeric|min:0',
            'daily_change' => 'required|numeric|min:0',
            'description' => 'nullable|string',
        ]);

        // Check if validation fails
        // if ($validator->fails()) {
        //     return response()->json([
        //         'status' => 'error',
        //         'errors' => $validator->errors()
        //     ], 422); // 422: Unprocessable Entity (used for validation errors)
        // }

        $exp = TradeExpert::findOrFail($id);
        $exp->name = $request->name;
        $exp->title = $request->title;
        $exp->status = $request->status;
        $exp->rating = $request->rating;
        $exp->followers = $request->followers;
        $exp->win_rate = $request->win_rate;
        $exp->total_return = $request->total_return;
        $exp->total_trades = $request->total_trades;
        $exp->min_amount = $request->min_amount;
        $exp->equity = $request->equity;
        $exp->daily_change = $request->daily_change;
        $exp->description = $request->description;

        if (!empty($request->file('image'))) {

            $file = $request->file('image');
            $randomStr = Str::random(30);
            $filename = $randomStr . '.' . $file->getClientOriginalExtension();
            $file->move('upload/experts/', $filename);
            $exp->image = $filename;
        }

        $exp->save();
        Log::info("TradeExpert updated successfully: ID {$exp->id}");

        return redirect()->route('admin.experts.index')
            ->with('success', 'Trade Expert Updated successfully!');
    }

    /**
     * Delete expert
     */
    public function destroy($id)
    {
        $expert = TradeExpert::findOrFail($id);

        if ($expert->image && Storage::disk('public')->exists($expert->image)) {
            Storage::disk('public')->delete($expert->image);
        }

        $expert->delete();

        return back()->with('success', 'Expert deleted successfully!');
    }


    public function updatedaily(Request $request, $id)
    {
        $expert = TradeExpert::findOrFail($id);

        $expert->update([
            'daily_change' => $request->daily_change,
            'status' => $request->status,
        ]);

        return back()->with('success', 'Expert trade status and daily change updated successfully!');
    }


    public function adminusercopytrades(Request $request)
    {
        $pageTitle = 'Users Copy Trading';

        $copy = UserCopyTrade::latest()->paginate(25);

        return view('admin/copy/trades', compact('pageTitle', 'copy'));
    }
}
