<?php

namespace App\Http\Controllers;

use App\Models\AiBot;
use App\Models\SignalPurchase;
use Illuminate\Support\Facades\Log;
use App\Models\WebsiteSetting;
use App\Models\BotTradeDailyRecord;
use Illuminate\Support\Facades\DB;
use App\Models\Withdrawal;
use App\Models\LiveTrade;
use App\Models\CopyTradeLog;
use App\Models\UserCopyTrade;
use App\Models\Investment;
use App\Models\Deposit;
use Illuminate\Support\Str;
use App\Models\User;
use App\Models\Currency;
use App\Models\Country;
use App\Models\InvestmentPlan;
use App\Models\UserBotTrade;
use Illuminate\Support\Facades\Auth;
use Illuminate\Http\Request;
use App\Models\ReferralLevel;
use App\Models\TradeExpert;
use App\Models\WalletProvider;
use Intervention\Image\Laravel\Facades\Image;
use Intervention\Image\Encoders\JpegEncoder;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use GuzzleHttp\Client;
use GuzzleHttp\Pool;
use PragmaRX\Google2FAQRCode\Google2FA;
use chillerlan\QRCode\QRCode;
use chillerlan\QRCode\QROptions;


class UserController extends Controller
{
    public function userdashboard(Request $request)
    {
        $pageTitle = 'Dashboard';

        $totalApprovedDeposit = Deposit::where('user_id', Auth::id())
            ->where('status', 'approved')
            ->sum('amount');

        $totalApprovedWithdraw = Withdrawal::where('user_id', Auth::id())
            ->where('status', 'approved')
            ->sum('amount');

        $livetrade = LiveTrade::where('user_id', Auth::id())
            ->latest()
            ->take(5)
            ->get();

        $purchases = SignalPurchase::where('user_id', auth()->id())->get();

        $invpurchases = Investment::where('user_id', auth()->id())
            ->where('status', 'active')
            ->get();

        return view('user/dashboard', compact('pageTitle', 'totalApprovedDeposit', 'livetrade', 'totalApprovedWithdraw', 'purchases', 'invpurchases'));
    }

    public function userprofile(Request $request)
    {
        $user = Auth::user();
        $pageTitle = 'Profile';

        $country = Country::where('status', '1')->orderBy('name', 'asc')->get();
        $countries = $user->country;

        $currency = Currency::where('status', '1')->get();
        $cur = $user->currency;

        $acctype = $user->account_type;

        $activity = Auth::user()
            ->activityLogs()
            ->latest()
            ->take(5)
            ->get();

        // Google Authenticator
        $google2fa = app('pragmarx.google2fa');

        $secret = $google2fa->generateSecretKey();

        $siteName = WebsiteSetting::value('site_name') ?? config('app.name');

        $qrUrl = $google2fa->getQRCodeUrl(
            $siteName,
            $user->email,
            $secret
        );

        $size = 220;
        $qrcodeUrl = 'https://quickchart.io/qr?text=' . urlencode($qrUrl) . '&size=220';

        return view('user.profile', compact(
            'pageTitle',
            'user',
            'secret',
            'qrcodeUrl',
            'country',
            'countries',
            'currency',
            'cur',
            'acctype',
            'activity'
        ));
    }

    public function updateuserprofile(Request $request)
    {
        $validated = $request->validate([
            'firstname' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'lastname' => ['required', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'account_type' => ['required', 'string', 'max:255'],
            'country' => ['nullable', 'string', 'max:255'],
            'phone' => ['nullable', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'address' => ['nullable', 'string', 'max:255', 'regex:/^[^<>]*$/'],
            'currency' => ['required', 'string', 'max:255'],
            'image' => 'nullable|image|mimes:jpg,png,jpeg|max:2048',
        ]);

        $user = Auth::user();

        // ✅ Default to existing image
        $filename = $user->image;

        // ✅ Process new image upload (if provided)
        if ($request->hasFile('image')) {
            $file = $request->file('image');

            // ✅ Generate random filename and always use .jpg
            $randomStr = Str::random(15);
            $filename = $randomStr . '.jpg';

            // ✅ Define destination path
            $destinationPath = public_path('upload/profile/');

            // ✅ Create folder if missing
            if (!file_exists($destinationPath)) {
                mkdir($destinationPath, 0755, true);
            }

            // ✅ Delete old image if it exists
            if ($user->image && file_exists($destinationPath . $user->image)) {
                @unlink($destinationPath . $user->image);
            }

            // ✅ Decode, sanitize, and re-encode to JPG (strips malicious code)
            $image = Image::read($file->getRealPath())
                ->orient()
                ->encode(new JpegEncoder(quality: 85));

            // ✅ Save safely
            $image->save($destinationPath . $filename);
        }

        // ✅ Update user details
        $user->firstname     = trim($validated['firstname']);
        $user->lastname     = trim($validated['lastname']);
        $user->account_type = trim($validated['account_type']);
        $user->country    = trim($validated['country']);
        $user->phone    = trim($validated['phone']);
        $user->address    = trim($validated['address']);
        $user->currency    = trim($validated['currency']);
        $user->image    = $filename;

        $user->save();

        // ✅ Log user activity (guaranteed call)
        if (function_exists('logUserActivity')) {
            logUserActivity('Profile Update', 'Your Profile was Updated.');
        } else {
            Log::warning('⚠️ logUserActivity() not found — ensure it’s globally loaded or autoloaded.');
        }

        return redirect()->back()->with('success', 'Profile updated successfully.');
    }

    public function showuseractivitylog(Request $request)
    {
        $pageTitle = 'Recent Activities';

        $activity = Auth::user()
            ->activityLogs()
            ->latest()
            ->get();

        return view('user/activity_log', compact('activity', 'pageTitle'));
    }

    public function showswapfund(Request $request)
    {
        $pageTitle = 'Recent Activities';

        return view('user/swap', compact('pageTitle'));
    }

    public function swapFunds(Request $request)
    {
        $validated = $request->validate([
            'from' => 'required|in:profit,bonus',
            'amount' => 'required|numeric|min:1',
        ]);

        $user = Auth::user();
        $from = $validated['from'];
        $amount = (float) $validated['amount'];

        // 💡 Dynamic balance source
        $sourceBalance = (float) $user->{$from};

        // 🚫 Check available funds
        if ($sourceBalance < $amount) {
            return back()->with('error', ucfirst($from) . " balance is insufficient for this transfer.");
        }

        // ✅ Perform swap safely
        DB::transaction(function () use ($user, $from, $amount) {
            // Deduct from selected wallet
            $user->decrement($from, $amount);

            // Credit to main balance
            $user->increment('balance', $amount);

            // Log user activity
            if (function_exists('logUserActivity')) {
                logUserActivity(
                    'Swap Funds',
                    "Transferred $" . number_format($amount, 2) . " from " . ucfirst($from) . " to Balance."
                );
            }
        });

        return back()->with('success', "Successfully transferred $" . number_format($amount, 2) . " from " . ucfirst($from) . " to your Balance.");
    }

    public function userstatement(Request $request)
    {
        $pageTitle = 'Account Statement';

        $user = Auth::user();

        // Optional search filter
        $search = $request->input('search');

        $deposits = Deposit::with('gateway') // eager-load payment gateway
            ->where('user_id', $user->id)
            ->when($search, function ($query, $search) {
                $query->where(function ($q) use ($search) {
                    $q->where('payment_mode', 'like', "%{$search}%")
                        ->orWhere('status', 'like', "%{$search}%")
                        ->orWhere('amount', 'like', "%{$search}%")
                        ->orWhereHas('gateway', function ($gatewayQuery) use ($search) {
                            $gatewayQuery->where('name', 'like', "%{$search}%");
                        });
                });
            })
            ->orderBy('created_at', 'desc')
            ->paginate(10);

        // Fetch user investments with related plan info
        $investments = Investment::with('plan')
            ->where('user_id', $user->id)
            ->orderBy('created_at', 'desc')
            ->get();

        $withdrawals = Withdrawal::where('user_id', Auth::id())->latest()->get();

        // Optionally calculate remaining days before passing to view
        foreach ($investments as $investment) {
            $investment->remaining_days = now()->diffInDays($investment->end_date, false);
        }

        return view('user/statement', compact('pageTitle', 'deposits', 'investments', 'withdrawals'));
    }

    public function userportfolio(Request $request)
    {
        $pageTitle = 'Portfolio';
        $user = Auth::user();

        // ✅ Fetch all investments by the authenticated user
        $investments = Investment::where('user_id', $user->id)
            ->latest()
            ->paginate(10);

        // ✅ Total number of investments
        $totalInvestments = Investment::where('user_id', $user->id)->count();

        // ✅ Count active investments
        $activeInvestments = Investment::where('user_id', $user->id)
            ->where('status', 'active')
            ->count();

        // ✅ Count completed investments
        $completedInvestments = Investment::where('user_id', $user->id)
            ->where('status', 'completed')
            ->count();

        // ✅ Sum total invested amount
        $totalAmountInvested = Investment::where('user_id', $user->id)->sum('amount');

        return view('user.portfolio', compact(
            'pageTitle',
            'investments',
            'totalInvestments',
            'activeInvestments',
            'completedInvestments',
            'totalAmountInvested'
        ));
    }


    public function userinvestmentdetails(Request $request, $id)
    {
        $pageTitle = 'Portfolio';

        $invest = Investment::with('logs', 'plan')->findOrFail($id);

        return view('user/investment_details', compact('pageTitle', 'invest'));
    }

    public function userinvestment(Request $request)
    {
        $user = Auth::user();

        $pageTitle = 'Investment';

        $invest = InvestmentPlan::where('status', 'Active')->get();

        if ($user->balance <= 0) {
            return redirect('user/deposit')->with('success', 'You need to make a deposit first.');
        }

        return view('user/investment', compact('pageTitle', 'invest'));
    }

    public function usermarket()
    {
        $pageTitle = 'Live Trade';
        $twelveKey = config('services.twelvedata.key');
        $vantageKey = config('services.vantage.key');

        // ============================================================
        // 1. CRYPTO (logos cached 1 day, prices live)
        // ============================================================

        try {
            // Cache coin info for 1 day
            $coinInfo = Cache::remember('coingecko_coin_info', 86400, function () {
                $list = Http::timeout(10)->get('https://api.coingecko.com/api/v3/coins/markets', [
                    'vs_currency' => 'usd',
                    'order' => 'market_cap_desc',
                    'per_page' => 50,
                    'page' => 1,
                    'sparkline' => false,
                ])->json();

                if (!is_array($list)) {
                    return [];
                }

                return collect($list)->mapWithKeys(function ($c) {
                    $symbol = strtoupper($c['symbol'] ?? '');
                    return [
                        $symbol => [
                            'name' => $c['name'] ?? $symbol,
                            'logo' => $c['image'] ?? null,
                        ],
                    ];
                });
            });

            // Live prices
            $liveData = Http::timeout(10)->get('https://api.coingecko.com/api/v3/coins/markets', [
                'vs_currency' => 'usd',
                'order' => 'market_cap_desc',
                'per_page' => 45,
                'page' => 1,
                'sparkline' => false,
            ])->json();

            if (!is_array($liveData)) {
                throw new \Exception('Invalid response from CoinGecko (live data)');
            }

            // Merge cached logos + live prices
            $crypto = collect($liveData)->map(function ($c) use ($coinInfo) {
                $symbol = strtoupper($c['symbol'] ?? '');
                $cached = $coinInfo[$symbol] ?? ['name' => $symbol, 'logo' => null];

                return [
                    'name' => $cached['name'],
                    'symbol' => $symbol,
                    'logo' => $cached['logo'],
                    'pair' => $symbol . '/USD',
                    'price' => $c['current_price'] ?? 0.00,
                    'change_24h' => $c['price_change_percentage_24h'] ?? 0.00,
                    'volume_24h' => $c['total_volume'] ?? 0.00,
                ];
            });
        } catch (\Exception $e) {
            // fallback
            $crypto = collect(range(1, 45))->map(fn($i) => [
                'name' => 'Crypto #' . $i,
                'symbol' => 'CRYPTO' . $i,
                'logo' => null,
                'pair' => 'CRYPTO' . $i . '/USD',
                'price' => 0.00,
                'change_24h' => 0.00,
                'volume_24h' => 0.00,
            ]);
        }

        // ============================================================
        // 2. STOCKS — Cached for 6 hours
        // ============================================================

        $stockSymbols = [
            'AAPL',
            'MSFT',
            'GOOGL',
            'AMZN',
            'TSLA',
            'NVDA',
            'META',
            'NFLX',
            'INTC',
            'AMD',
            'IBM',
            'ORCL',
            'CSCO',
            'ADBE',
            'CRM',
            'PYPL',
            'QCOM',
            'TXN',
            'AVGO',
            'SAP',
            'SHOP',
            'UBER',
            'LYFT',
            'SQ',
            'TWTR',
            'SNOW'
        ];

        $clearbitMapping = [
            'AAPL' => 'apple.com',
            'MSFT' => 'microsoft.com',
            'GOOGL' => 'abc.xyz',
            'AMZN' => 'amazon.com',
            'TSLA' => 'tesla.com',
            'NVDA' => 'nvidia.com',
            'META' => 'meta.com',
            'NFLX' => 'netflix.com',
            'INTC' => 'intel.com',
            'AMD' => 'amd.com',
            'IBM' => 'ibm.com',
            'ORCL' => 'oracle.com',
            'CSCO' => 'cisco.com',
            'ADBE' => 'adobe.com',
            'CRM' => 'salesforce.com',
            'PYPL' => 'paypal.com',
            'QCOM' => 'qualcomm.com',
            'TXN' => 'ti.com',
            'AVGO' => 'broadcom.com',
            'SAP' => 'sap.com',
            'SHOP' => 'shopify.com',
            'UBER' => 'uber.com',
            'LYFT' => 'lyft.com',
            'SQ' => 'block.xyz',
            'TWTR' => 'twitter.com',
            'SNOW' => 'snowflake.com'
        ];

        // Cache for 6 hours (21600 seconds)
        $stocks = Cache::remember('stocks_data', 21600, function () use ($stockSymbols, $clearbitMapping, $twelveKey) {

            return collect($stockSymbols)->map(function ($symbol) use ($clearbitMapping, $twelveKey) {
                try {
                    $response = Http::timeout(10)->get('https://api.twelvedata.com/quote', [
                        'symbol' => $symbol,
                        'exchange' => 'NASDAQ',
                        'apikey' => $twelveKey,
                    ])->json();

                    $name = $response['name'] ?? $symbol;
                    $price = isset($response['close']) ? (float) $response['close'] : null;
                    $percentChange = isset($response['percent_change']) ? (float) $response['percent_change'] : null;

                    $domain = $clearbitMapping[$symbol] ?? null;
                    $logo = $domain
                        ? "https://img.logo.dev/{$domain}?token=" . config('services.logodev.publishable_key')
                        : asset('images/default-logo.png');

                    return [
                        'name' => $name,
                        'symbol' => $symbol,
                        'pair' => $symbol,
                        'logo' => $logo,
                        'price' => $price ? number_format($price, 2) : '0.00',
                        'change_24h' => $percentChange !== null ? number_format($percentChange, 2) . '%' : 'N/A',
                    ];
                } catch (\Exception $e) {
                    return [
                        'name' => $symbol,
                        'symbol' => $symbol,
                        'pair' => $symbol,
                        'logo' => asset('images/default-logo.png'),
                        'price' => '0.00',
                        'change_24h' => 'N/A',
                    ];
                }
            });
        });

        // ============================================================
        // 3. FOREX — Cached for 6 hours
        // ============================================================

        $forexPairs = [
            'EUR/USD',
            'GBP/USD',
            'USD/JPY',
            'AUD/USD',
            'NZD/USD',
            'USD/CAD',
            'USD/CHF',
            'USD/CNY',
            'USD/SEK',
            'USD/NOK',
            'USD/DKK',
            'USD/SGD',
            'USD/HKD',
            'USD/INR',
            'USD/KRW',
            'USD/MXN',
            'USD/BRL',
            'USD/ZAR',
            'USD/RUB',
            'USD/TRY',
        ];

        // logo cache
        $logos = Cache::remember('forex_logos', 86400, function () use ($forexPairs) {
            return collect($forexPairs)->mapWithKeys(function ($pair) {
                [$base] = explode('/', $pair);
                $flag = strtoupper(substr($base, 0, 2));
                return [$pair => "https://flagsapi.com/{$flag}/flat/64.png"];
            });
        });

        $forex = Cache::remember('forex_data', 21600, function () use ($forexPairs, $logos, $twelveKey) {

            $collection = collect();

            foreach ($forexPairs as $pair) {
                try {
                    $response = Http::timeout(10)->get('https://api.twelvedata.com/quote', [
                        'symbol' => $pair,
                        'apikey' => $twelveKey,
                    ])->json();

                    $price = $response['close'] ?? 0;
                    $change = $response['percent_change'] ?? 0;

                    $collection->push([
                        'name' => $response['name'] ?? $pair,
                        'symbol' => str_replace('/', '', $pair),
                        'pair' => $pair,
                        'logo' => $logos[$pair],
                        'price' => number_format($price, 5),
                        'change_24h' => number_format($change, 2),
                    ]);
                } catch (\Exception $e) {
                    $collection->push([
                        'name' => $pair,
                        'symbol' => str_replace('/', '', $pair),
                        'pair' => $pair,
                        'logo' => $logos[$pair],
                        'price' => 0,
                        'change_24h' => 0,
                    ]);
                }
            }

            return $collection;
        });

        // ============================================================
        // 4. COMMODITIES — Cached 6 hours
        // ============================================================

        $commodities = Cache::remember('commodities_data', 21600, function () use ($vantageKey) {

            $commodityMap = [
                "CL1" => "WTI",
                "NG/USD" => "NG",
                "HG1" => "HG",
                "LMAHDS03" => "AL",
                "W_1" => "WHEAT",
                "C_1" => "CORN",
                "CT1" => "CT",
                "SB1" => "SB",
                "KC1" => "KC"
            ];

            return collect($commodityMap)->map(function ($function, $symbol) use ($vantageKey) {
                $price = 0;

                try {
                    $response = Http::timeout(10)->get('https://www.alphavantage.co/query', [
                        'function' => $function,
                        'interval' => 'daily',
                        'apikey' => $vantageKey
                    ])->json();

                    $latest = $response['data'][0] ?? null;
                    $price = $latest['value'] ?? 0;
                } catch (\Exception $e) {
                }

                return [
                    'symbol' => $symbol,
                    'name' => $function,
                    'pair' => $symbol,
                    'price' => number_format($price, 2),
                    'change_24h' => '0.00',
                    'logo' => 'https://cdn-icons-png.flaticon.com/512/190/190411.png',
                ];
            });
        });

        // ============================================================
        // 5. BONDS — Cached 6 hours
        // ============================================================

        $bonds = Cache::remember('bonds_data', 21600, function () {
            $symbols = ['US10Y', 'US30Y', 'US5Y', 'US2Y', 'US1Y', 'US3M', 'US6M', 'US1M', 'US7Y', 'US20Y'];

            return collect($symbols)->map(fn($symbol) => [
                'name' => $symbol,
                'symbol' => $symbol,
                'pair' => $symbol,
                'logo' => strtoupper(substr($symbol, 0, 2)),
                'price' => '0.00',
                'change_24h' => '0.00%',
            ]);
        });

        // ============================================================
        return view('user/market', compact('pageTitle', 'crypto', 'stocks', 'forex', 'commodities', 'bonds'));
    }


    public function usertrade(Request $request)
    {
        $data = [
            'symbol' => $request->query('symbol'),
            'type' => ucfirst($request->query('type')),
            'name' => $request->query('name'),
            'price' => $request->query('price'),
            'pair' => $request->query('pair'),
            'logo' => $request->query('logo'),
            'change_24h' => $request->query('change_24h'),
        ];

        $pageTitle = "Trade " . strtoupper($data['symbol']);

        return view('user/trade', compact('data', 'pageTitle'));
    }

    public function userlivetrade(Request $request)
    {
        $openTrades = LiveTrade::where('status', 'open')
            ->where('user_id', auth()->id())
            ->latest()
            ->get();

        $closedTrades = LiveTrade::whereIn('status', ['closed', 'completed'])
            ->where('user_id', auth()->id())
            ->latest()
            ->get();

        $pageTitle = 'Market Orders';

        return view('user/live_trades', compact('pageTitle', 'openTrades', 'closedTrades'));
    }

    public function usermonitortrade(Request $request, $id)
    {
        $trade = LiveTrade::findorfail($id);

        $pageTitle = 'Monitor ' . $trade->asset_name . ' trade';

        $userTrades = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->latest()
            ->get();

        $tradeCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->get();

        $activeCount = LiveTrade::where('user_id', auth()->id())
            ->where('status', 'open')
            ->where('symbol', $trade->symbol)
            ->count();

        $completeCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->where('status', 'completed')
            ->count();

        $closedCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->where('status', 'closed')
            ->count();

        $totalInvested = $tradeCount->sum('amount');

        $totalTrades = $tradeCount->count();

        return view('user/monitor_trade', compact('trade', 'pageTitle', 'userTrades', 'closedCount', 'tradeCount', 'totalTrades', 'totalInvested', 'activeCount', 'completeCount'));
    }

    public function usertradedetails(Request $request, $id)
    {
        $trade = LiveTrade::findorfail($id);

        $pageTitle = 'View ' . $trade->asset_name . ' trade details';

        $userTrades = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->latest()
            ->get();

        $tradeCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->get();

        $activeCount = LiveTrade::where('user_id', auth()->id())
            ->where('status', 'open')
            ->where('symbol', $trade->symbol)
            ->count();

        $completeCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->where('status', 'completed')
            ->count();

        $closedCount = LiveTrade::where('user_id', auth()->id())
            ->where('symbol', $trade->symbol)
            ->where('status', 'closed')
            ->count();

        $totalInvested = $tradeCount->sum('amount');

        $totalTrades = $tradeCount->count();

        return view('user/trade_details', compact('trade', 'pageTitle', 'userTrades', 'closedCount', 'tradeCount', 'totalTrades', 'totalInvested', 'activeCount', 'completeCount'));
    }

    public function userbottrade(Request $request)
    {
        $pageTitle = 'AI Trade';

        $bots = AiBot::inRandomOrder()->get();

        $forexbots = AiBot::where('trading_type', 'forex')->inRandomOrder()->get();

        $cryptobots = AiBot::where('trading_type', 'crypto')->inRandomOrder()->get();

        $stockbots = AiBot::where('trading_type', 'stock')->inRandomOrder()->get();

        return view('user/bot_ai_trading', compact('pageTitle', 'bots', 'forexbots', 'cryptobots', 'stockbots'));
    }

    public function userbotdashboard(Request $request)
    {
        $user = Auth::user();

        $pageTitle = 'Bot Trading Dashboard';

        // --- 1️⃣ Fetch user’s bot trades
        $trades = UserBotTrade::with('bot')
            ->where('user_id', $user->id)
            ->orderByDesc('created_at')
            ->get();

        $runningtrades = UserBotTrade::with('bot')
            ->where('user_id', $user->id)
            ->where('status', 'running')
            ->latest()->get();

        // 🧠 Fetch all daily records for the user through their trades
        $dailyLogs = BotTradeDailyRecord::whereHas('trade', function ($query) use ($user) {
            $query->where('user_id', $user->id);
        })
            ->with(['trade.bot']) // eager load related bot and trade info
            ->latest()
            ->take(12)->get();

        // --- 2️⃣ Compute key stats
        $activeTrades = $trades->where('status', 'running');
        $completedTrades = $trades->where('status', 'completed');

        $activeCount = $activeTrades->count();
        $completedCount = $completedTrades->count();
        $totalTrades = $trades->count();

        // --- 3️⃣ Profit & loss breakdown
        $totalProfit = $trades->sum('profit_earned');
        $totalInvested = $trades->sum('amount_invested');

        $currentBalance = $trades->where('user_id', $user->id)
            ->where('status', 'running')
            ->sum('current_value');

        // If you want to separate positive vs negative profits:
        $totalPositive = $trades->where('profit_earned', '>', 0)->sum('profit_earned');
        $totalNegative = $trades->where('profit_earned', '<', 0)->sum('profit_earned');

        // --- 4️⃣ Current balance (assuming a "balance" field exists on users table)
        // $currentBalance = $user->balance ?? 0;

        // --- 5️⃣ Top performing bots (by profit)
        $topBots = $trades->groupBy('ai_bot_id')->map(function ($botTrades) {
            return [
                'bot' => $botTrades->first()->bot->name ?? 'Unknown Bot',
                'total_profit' => $botTrades->sum('profit_earned'),
                'trades' => $botTrades->count(),
            ];
        })->sortByDesc('total_profit')->take(5);

        // --- 6️⃣ Data for charts (optional)
        $profitChartData = $trades->sortBy('created_at')->map(function ($trade) {
            return [
                'date' => $trade->created_at->format('Y-m-d'),
                'profit' => $trade->profit_earned,
            ];
        })->values();

        // --- 7️⃣ Return to blade
        return view('user.bot_dashboard', compact(
            'pageTitle',
            'user',
            'trades',
            'activeCount',
            'completedCount',
            'totalTrades',
            'totalProfit',
            'totalPositive',
            'totalNegative',
            'totalInvested',
            'currentBalance',
            'topBots',
            'profitChartData',
            'runningtrades',
            'dailyLogs'
        ));
    }


    public function usercopytrade(Request $request)
    {
        $pageTitle = 'Copy Experts';
        $user = Auth::user();

        // Fetch copied experts
        $copiedExperts = UserCopyTrade::with('expert')
            ->where('user_id', $user->id)
            ->where('status', 'active')
            ->get()
            ->map(function ($item) use ($user) {
                // Sum of today's profits for this expert
                $dailyProfit = CopyTradeLog::where('user_id', $user->id)
                    ->where('trade_expert_id', $item->trade_expert_id)
                    ->whereDate('created_at', now()->toDateString())
                    ->sum('profit_amount');

                $item->daily_profit = $dailyProfit;
                return $item;
            });

        // Count active copy trades
        $activeCount = $copiedExperts->count();

        // --- New Calculations ---

        // 1️⃣ Total invested amount (sum of user investment per copy trade)
        $totalInvested = $copiedExperts->sum('amount');

        // 2️⃣ Total current value (investment + total profit so far for each copy)
        $totalCurrentValue = $copiedExperts->sum(function ($item) use ($user) {
            $totalProfit = CopyTradeLog::where('user_id', $user->id)
                ->where('trade_expert_id', $item->trade_expert_id)
                ->sum('profit_amount');

            return $item->amount + $totalProfit;
        });

        // 3️⃣ Total P&L (Profit & Loss)
        $totalPnL = $totalCurrentValue - $totalInvested;

        $totalROI = $copiedExperts->sum('roi_percentage');

        // Determine P&L sign (positive or negative)
        $pnlSign = $totalPnL >= 0 ? 'positive' : 'negative';

        $tradelog = UserCopyTrade::where('status', 'stopped')->get();

        return view('user/copy_trading', compact(
            'pageTitle',
            'copiedExperts',
            'activeCount',
            'totalInvested',
            'totalCurrentValue',
            'totalPnL',
            'pnlSign',
            'totalROI',
            'tradelog'
        ));
    }

    // public function usercopytradedetails(Request $request, $id)
    // {
    //     return view('user/copy_tradedetails');
    // }


    public function usercopyexperts(Request $request)
    {
        $pageTitle = 'Browse Experts';

        $exp = TradeExpert::all();

        $totalexp = TradeExpert::count();

        return view('user/copy_experts', compact('pageTitle', 'exp', 'totalexp'));
    }

    public function userconnectwallet(Request $request)
    {
        $pageTitle = 'Connect Wallet';

        // orderBy('name', 'asc')->get();

        $wallets = WalletProvider::all();

        return view('user/connectwallet', compact('pageTitle', 'wallets'));
    }

    public function userreferrals(Request $request)
    {
        $pageTitle = 'Referral Program';

        $levels = ReferralLevel::all();

        return view('user/referrals', compact('pageTitle', 'levels'));
    }


    public function userkycindex(Request $request)
    {
        $pageTitle = 'Kyc Verification';

        return view('user/verify_account', compact('pageTitle'));
    }

    public function userkycstart(Request $request)
    {
        $pageTitle = 'Kyc Verification';

        $country = Country::where('status', '1')
            ->orderBy('name', 'asc')
            ->get();

        return view('user/start_verification', compact('pageTitle', 'country'));
    }


    public function usersupport(Request $request)
    {
        return view('user/support');
    }
}
